# btc4096.py
# Generates fully automated 4096-vector HDGL covenant script
# Recursively folds 64-vector blocks with safe saturation
# Only unique r_dim & Ω embedded per block

MAX_VECTORS = 4096
BLOCK_SIZE = 64  # fold 64 vectors at a time
SAFE_ZERO = "0x0"

# Example r_dim & Ω per 64-vector block (powers-of-2 scaled for safety)
# Only unique nonzero params embedded
PARAMS = {
    1: ("0x3000", "0x812"),
    2: ("0x4000", "0x502"),
    3: ("0x5000", "0x310"),
    4: ("0x6000", "0x192"),
    5: ("0x7000", "0x119"),
    6: ("0x8000", "0x73"),
    7: ("0x9000", "0x45"),
    8: ("0xA000", "0x28"),
    # repeat last param for remaining blocks if needed
}

def generate_block(block_idx, idx_base):
    """Generate a single 64-vector block"""
    script = []

    # generate vectors 0..63 (compressed zeros)
    v_line = " ".join([SAFE_ZERO]*4) + " OP_HDGL_VEC_ADD OP_EQUALVERIFY"
    for i in range(BLOCK_SIZE // 4):
        script.append(v_line)

        add_idx = idx_base + i*4
        add_line = f"{add_idx} {add_idx+1} OP_ADD {add_idx+2} OP_EQUALVERIFY"
        within_line = f"{add_idx+3} {add_idx+4} {add_idx+5} OP_WITHINVERIFY"
        script.append(add_line)
        script.append(within_line)

        # embed unique r_dim & Ω params if defined
        if block_idx in PARAMS and i == 0:  # embed once per block
            param_line = f"{PARAMS[block_idx][0]} {PARAMS[block_idx][1]} OP_HDGL_PARAM_ADD OP_EQUALVERIFY"
            script.append(param_line)

        script.append("")  # spacing

    return script

def generate_hdgl_covenant(pubkey="<YOUR_PUBLIC_KEY>"):
    script = [f"{pubkey} OP_CHECKSIGVERIFY\n"]

    num_blocks = MAX_VECTORS // BLOCK_SIZE
    idx_base = 100
    for block_idx in range(1, num_blocks + 1):
        block_script = generate_block(block_idx, idx_base)
        script.extend(block_script)
        idx_base += BLOCK_SIZE  # increment base index for next block

    script.append("OP_CHECKTEMPLATEVERIFY")
    return "\n".join(script)

if __name__ == "__main__":
    covenant_script = generate_hdgl_covenant()
    print("=== Generated 4096-Vector HDGL Covenant ===")
    print(covenant_script)
